<?php

namespace AminulDigital\Extensions;

class Manager
{
    /**
     * @var string
     */
    protected string $dir;

    /**
     * @var array
     */
    private const EXTENSIONS_KEYS = [
        'AVAILABLE' => 'available',
        'UNAVAILABLE' => 'unavailable',
    ];

    private array $extensions = [
        self::EXTENSIONS_KEYS['AVAILABLE'] => [],
        self::EXTENSIONS_KEYS['UNAVAILABLE'] => [],
    ];

    public function __construct(?string $dir = null)
    {
        $this->dir = trim($dir ?? __DIR__ . DIRECTORY_SEPARATOR . 'extensions');
        $this->loadExtensions();
    }

    /**
     * Get all extensions.
     *
     * @return array
     */
    public function all(): array
    {
        return $this->extensions;
    }

    /**
     * Get available extensions.
     *
     * @return array
     */
    public function available(): array
    {
        return $this->extensions[self::EXTENSIONS_KEYS['AVAILABLE']];
    }

    /**
     * Get unavailable extensions.
     *
     * @return array
     */
    public function unavailable(): array
    {
        return $this->extensions[self::EXTENSIONS_KEYS['UNAVAILABLE']];
    }

    /**
     * Load extensions.
     *
     * @return void
     */
    private function loadExtensions(): void
    {
        foreach ($this->findExtensions() as $extension) {
            try {
                $ext = require_once $extension;
                $ext['root'] = dirname($extension);

                $this->extensions[self::EXTENSIONS_KEYS['AVAILABLE']][$extension] = $ext;
            } catch (\Throwable $e) {
                $this->extensions[self::EXTENSIONS_KEYS['UNAVAILABLE']][$extension] = [
                    'name' => basename($extension),
                    'error' => $e->getMessage(),
                ];
            }
        }
    }

    /**
     * Find extensions.
     *
     * @return array
     */
    private function findExtensions(): array
    {
        if (!is_dir($this->dir)) {
            return [];
        }

        $extensions = [];

        $paths = array_diff(scandir($this->dir), ['..', '.']);
        foreach ($paths as $path) {
            if (is_dir($this->dir . DIRECTORY_SEPARATOR . $path)) {
                $extensions[] = $this->dir . DIRECTORY_SEPARATOR . $path . DIRECTORY_SEPARATOR . $path . '.php';
            } else {
                $extensions[] = $this->dir . DIRECTORY_SEPARATOR . $path;
            }
        }

        return $extensions;
    }
}
