<?php

namespace AminulDigital\Extensions;

class Loader
{
    /**
     * @var array
     */
    private array $prefixes = [];

    /**
     * @var array
     */
    private array $files = [];

    /**
     * Add prefix and path
     *
     * @param string $prefix
     * @param string $path
     *
     * @return void
     */
    public function add(string $prefix, string $path): void
    {
        $this->prefixes[] = [
            'prefix' => rtrim($prefix, '\\') . '\\',
            'path' => $path,
        ];
    }

    /**
     * Add static file
     *
     * @param string $file
     *
     * @return void
     */
    public function addFile(string $file): void
    {
        $this->files[] = $file;
    }

    /**
     * Load classes
     *
     * @return void
     */
    public function load(): void
    {
        $this->requireFiles();
        $this->registerAutoload();
    }

    /**
     * Require files
     *
     * @return void
     */
    private function requireFiles(): void
    {
        foreach ($this->files as $file) {
            require_once $file;
        }
    }

    /**
     * Register autoload
     *
     * @return void
     */
    private function registerAutoload(): void
    {
        spl_autoload_register([$this, 'autoload']);
    }

    /**
     * Autoload classes
     *
     * @param string $class
     *
     * @return void
     */
    public function autoload(string $class): void
    {
        if ($file = $this->findFile($class)) {
            require_once $file;
        }
    }

    /**
     * Find file
     *
     * @param string $class
     *
     * @return string|null
     */
    private function findFile(string $class): ?string
    {
        foreach ($this->prefixes as $prefix) {
            $len = strlen($prefix['prefix']);
            if (strncmp($prefix['prefix'], $class, $len) !== 0) {
                continue;
            }

            $relative_class = substr($class, $len);
            $file = $prefix['path'] . DIRECTORY_SEPARATOR . str_replace('\\', '/', $relative_class) . '.php';

            if (file_exists($file)) {
                return $file;
            }
        }

        return null;
    }
}
