<?php

namespace AminulDigital\Extensions;

use Illuminate\Support\ServiceProvider;

class ExtensionsServiceProvider extends ServiceProvider
{
    /**
     * Available extensions.
     *
     * @var array
     */
    private array $extensions = [];

    /**
     * Register services.
     *
     * @return void
     */
    public function register(): void
    {
        $this->loadExtensions();
        $this->registerExtensions();
    }

    /**
     * Bootstrap services.
     *
     * @return void
     */
    public function boot(): void
    {
        $this->bootExtensions();
    }

    /**
     * Load available extensions.
     *
     * @return void
     */
    private function loadExtensions(): void
    {
        $dir = base_path('extensions');
        $manager = new Manager($dir);
        $this->extensions = $manager->available();
    }

    /**
     * Register available extensions.
     *
     * @return void
     */
    private function registerExtensions(): void
    {
        $loader = new Loader();

        foreach ($this->extensions as $ext) {
            $this->registerNamespaces($loader, $ext);
            $this->registerFiles($loader, $ext);
            $this->registerExtension($ext);
        }

        $loader->load();
    }

    /**
     * Register namespaces for an extension.
     *
     * @param \AminulDigital\Extensions\Loader $loader
     * @param array $ext
     *
     * @return void
     */
    private function registerNamespaces(Loader $loader, array $ext): void
    {
        $libs = data_get($ext, 'libs.namespaces', []);

        foreach ($libs as $path => $namespace) {
            $loader->add($namespace, $ext['root'] . DIRECTORY_SEPARATOR . $path);
        }
    }

    /**
     * Register files for an extension.
     *
     * @param \AminulDigital\Extensions\Loader $loader
     * @param array $ext
     *
     * @return void
     */
    private function registerFiles(Loader $loader, array $ext): void
    {
        $files = data_get($ext, 'libs.files', []);

        foreach ($files as $file) {
            $file = $ext['root'] . DIRECTORY_SEPARATOR . ltrim($file, DIRECTORY_SEPARATOR);
            $loader->addFile($file);
        }
    }

    /**
     * Register an extension.
     *
     * @param array $ext
     *
     * @return void
     */
    private function registerExtension(array $ext): void
    {
        if (isset($ext['register']) && is_callable($ext['register'])) {
            $ext['register']($this->app);
        }
    }

    /**
     * Bootstrap extensions.
     *
     * @return void
     */
    private function bootExtensions(): void
    {
        foreach ($this->extensions as $ext) {
            $this->bootExtension($ext);
        }
    }

    /**
     * Bootstrap an extension.
     *
     * @param array $ext
     *
     * @return void
     */
    private function bootExtension(array $ext): void
    {
        if (isset($ext['boot']) && is_callable($ext['boot'])) {
            $ext['boot']($this->app);
        }
    }
}
